'use strict';

/* --------------------------------------------------------------
 categories_multi_action_controller.js 2018-04-18
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Orders Table Controller
 *
 * This controller contains the mapping logic of the categories/articles multi select actions for the button
 * dropdown (on the bottom).
 *
 * @module Compatibility/categories_multi_action_controller
 */
gx.compatibility.module('categories_multi_action_controller', [gx.source + '/libs/button_dropdown'],

/**  @lends module:Compatibility/categories_multi_action_controller */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {},


    /**
     * Dropdown button selector
     * @var {object}
     */
    $dropdown = $this.find('.js-bottom-dropdown'),


    /**
     * Input fields
     * @type {*|jQuery|HTMLElement}
     */
    $inputs = $('tr[data-id] input[type="checkbox"]'),


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {};

    // ------------------------------------------------------------------------
    // PRIVATE METHODS
    // ------------------------------------------------------------------------


    /**
     * Get Url Parameter
     *
     * Gets a specific URL get parameter from the address bar,
     * which name should be provided as an argument.
     * @param {string} parameterName
     * @returns {object}
     * @private
     */
    var _getUrlParameter = function _getUrlParameter(parameterName) {
        var results = new RegExp('[\?&]' + parameterName + '=([^&#]*)').exec(window.location.href);
        if (results == null) {
            return null;
        } else {
            return results[1] || 0;
        }
    };

    /**
     * Prepare Form
     *
     * @param {string} action
     *
     * @return {object | jQuery}
     */
    var _$prepareForm = function _$prepareForm(action) {
        var cPath;
        try {
            cPath = window.location.href.match(/cPath=(.*)/)[1];
        } catch (e) {
            cPath = $('[data-cpath]:first').data().cpath;
        }

        var page_token = $('input[name="page_token"]:first').attr('value');

        var formUrl = [_getSourcePath(), 'categories.php', '?action=multi_action', '&cPath=' + cPath].join('');

        var search = _getUrlParameter('search');
        if (search !== 0 && search !== null) {
            formUrl += '&search=' + search;
        }

        var page = _getUrlParameter('page');
        if (page !== 0 && page !== null && formUrl.indexOf('page=') === -1) {
            formUrl += '&page=' + page;
        }

        var sorting = _getUrlParameter('sorting');
        if (sorting !== 0 && sorting !== null) {
            formUrl += '&sorting=' + sorting;
        }

        var $form = $('<form name="multi_action_form" method="post" action=' + formUrl + '></form>');
        $form.append('<input type="hidden" name="cpath" value=' + cPath + '>');
        $form.append('<input type="hidden" name="page_token" value=' + page_token + '>');
        $form.append('<input type="hidden" name=' + action + ' value="Action">');
        $form.appendTo('body');
        return $form;
    };

    var _sectionMapping = {
        delete: 'buttons',
        BUTTON_MOVE: 'admin_buttons',
        BUTTON_COPY: 'admin_buttons',
        BUTTON_STATUS_ON: 'admin_buttons',
        BUTTON_STATUS_OFF: 'admin_buttons'
    };

    /**
     * Map actions for the dropdown button
     *
     * This method will map the actions for multiple selects.
     */
    var _mapMultiActions = function _mapMultiActions() {
        var actions = ['delete', 'BUTTON_MOVE', 'BUTTON_COPY', 'BUTTON_STATUS_ON', 'BUTTON_STATUS_OFF'];

        for (var index in actions) {
            _mapAction(actions[index]);
        }
    };

    var _mapAction = function _mapAction(action) {
        var section = _sectionMapping[action],
            callback = _getActionCallback(action);
        jse.libs.button_dropdown.mapAction($dropdown, action, section, callback);
    };

    var _callbackDelete = function _callbackDelete(event) {
        // Do not do anything when no product/category is checked
        if (!$inputs.filter(':checked').length) {
            return;
        }

        // Submit cached form
        var $form = _$prepareForm('multi_delete');
        $inputs.filter(':checked').appendTo($form);
        $form.submit();
    };

    var _callbackMove = function _callbackMove(event) {
        // Do not do anything when no product/category is checked
        if (!$inputs.filter(':checked').length) {
            return;
        }

        // Submit cached form
        var $form = _$prepareForm('multi_move');
        $inputs.filter(':checked').appendTo($form);
        $form.submit();
    };

    var _callbackCopy = function _callbackCopy(event) {
        // Do not do anything when no product/category is checked
        if (!$inputs.filter(':checked').length) {
            return;
        }

        // Submit cached form
        var $form = _$prepareForm('multi_copy');
        $inputs.filter(':checked').appendTo($form);
        $form.submit();
    };

    var _callbackStatusOn = function _callbackStatusOn(event) {
        // Do not do anything when no product/category is checked
        if (!$inputs.filter(':checked').length) {
            return;
        }

        // Submit cached form
        var $form = _$prepareForm('multi_status_on');
        $inputs.filter(':checked').appendTo($form);
        $form.submit();
    };

    var _callbackStatusOff = function _callbackStatusOff(event) {
        // Do not do anything when no product/category is checked
        if (!$inputs.filter(':checked').length) {
            return;
        }

        // Submit cached form
        var $form = _$prepareForm('multi_status_off');
        $inputs.filter(':checked').appendTo($form);
        $form.submit();
    };

    var _getActionCallback = function _getActionCallback(action) {
        switch (action) {
            case 'delete':
                return _callbackDelete;
            case 'BUTTON_MOVE':
                return _callbackMove;
            case 'BUTTON_COPY':
                return _callbackCopy;
            case 'BUTTON_STATUS_ON':
                return _callbackStatusOn;
            case 'BUTTON_STATUS_OFF':
                return _callbackStatusOff;
            default:
                console.alert('_getActionCallback: Action not found');
        }
        return null;
    };

    /**
     * Get path of the admin folder
     *
     * @returns {string}
     */
    var _getSourcePath = function _getSourcePath() {
        var url = window.location.origin,
            path = window.location.pathname;

        var splittedPath = path.split('/');
        splittedPath.pop();

        var joinedPath = splittedPath.join('/');

        return url + joinedPath + '/';
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Wait until the dropdown is filled
        var interval = setInterval(function () {
            if ($('.js-button-dropdown').length > 0) {
                clearInterval(interval);
                _mapMultiActions();
            }
        }, 200);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
